# Simulate coalescent process with heterochronous sampling times

# Assumptions and modifications
# - bottlenecks of increasingly large drop considered
# - uses phylodyn package of Karcher 2016 et al
# - samples placed uniformly across bottleneck

# Clean the workspace and console
closeAllConnections()
rm(list=ls())
cat("\014")  
graphics.off()

# Packages for phylodyn
library("sp")
library("devtools")
library("INLA")
library("spam")
library("ape")
library("phylodyn")

# Set working directory to source
this.dir <- dirname(parent.frame(2)$ofile)
setwd(this.dir)

# Function to write simple csv files to correct path
tableWrite <- function(val, name, pathname) {
  # Add path to name
  str0 <- paste(c(pathname, name), collapse = "")
  # Write table
  write.table(val, str0, row.names=FALSE, col.names=FALSE, sep=",")
}


# Main code for heterochronous simulations ----------------------------------------------------------

# Possible trajectories
fracs = seq(0.05, 0.5, length.out = 10)
num = length(fracs)

for (i in 1:num) {
  # Set population true trajectory
  trajName = paste(c('bottle', i), collapse = '')
  dir.create(file.path(this.dir, trajName))

  # Define a middling bottleneck with amplitude
  bottle_traj <- function (t, frac = fracs[i]) 
  {
    result = rep(0, length(t))
    result[t <= 50] <- 1000
    result[t > 50 & t < 150] <- frac*1000
    result[t >= 150] <- 1000
    return(result)
  }
  traj = bottle_traj
  
  # Set sampling interval end and no. samples
  all_samp_end = 200
  nsamps = 500; ndivs = 20
  # Sample number and times
  samps = rep(nsamps/ndivs, ndivs)
  samp_times = seq(0, all_samp_end, length.out = ndivs)
  
  # Simulate genealogy and get all times
  gene = coalsim(samp_times = samp_times, n_sampled = samps, traj = traj, lower_bound = 10, method = "thin")
  coal_times = gene$coal_times
  coalLin = gene$lineages
  
  # Obtain true trajectory across time
  tmax = max(coal_times)
  t = seq(0, tmax, length=40000)
  y = traj(t)
  # Plot trajectory
  #quartz()
  #plot(t, y)
  
  # Export data for Matlab
  pathf = paste(c(this.dir, '/', trajName, '/'), collapse = "")
  tableWrite(coal_times, 'coaltimes.csv', pathf)
  tableWrite(samp_times, 'samptimes.csv', pathf)
  tableWrite(coalLin, 'coalLin.csv', pathf)
  tableWrite(y, 'trajy.csv', pathf)
  tableWrite(t, 'trajt.csv', pathf)
  tableWrite(samps, 'sampIntro.csv', pathf)
  
  # Plot and write tree
  tree <-generate_newick(gene)
  #quartz()
  #plot(tree$newick, show.tip.label = F)
  currDir = this.dir
  setwd(file.path(this.dir, trajName))
  write.tree(tree$newick, file="tree.txt")
  setwd(currDir)
}
